/*****************************************************************************

 @(#) sad.c,v openss7-0_9_2_F(0.9.2.40) 2007/03/25 19:02:29

 -----------------------------------------------------------------------------

 Copyright (c) 2001-2006  OpenSS7 Corporation <http://www.openss7.com/>
 Copyright (c) 1997-2000  Brian F. G. Bidulock <bidulock@openss7.org>

 All Rights Reserved.

 This program is free software; you can redistribute it and/or modify it under
 the terms of the GNU General Public License as published by the Free Software
 Foundation; version 2 of the License.

 This program is distributed in the hope that it will be useful, but WITHOUT
 ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or FITNESS
 FOR A PARTICULAR PURPOSE.  See the GNU General Public License for more
 details.

 You should have received a copy of the GNU General Public License along with
 this program; if not, write to the Free Software Foundation, Inc., 675 Mass
 Ave, Cambridge, MA 02139, USA.

 -----------------------------------------------------------------------------

 U.S. GOVERNMENT RESTRICTED RIGHTS.  If you are licensing this Software on
 behalf of the U.S. Government ("Government"), the following provisions apply
 to you.  If the Software is supplied by the Department of Defense ("DoD"), it
 is classified as "Commercial Computer Software" under paragraph 252.227-7014
 of the DoD Supplement to the Federal Acquisition Regulations ("DFARS") (or any
 successor regulations) and the Government is acquiring only the license rights
 granted herein (the license rights customarily provided to non-Government
 users).  If the Software is supplied to any unit or agency of the Government
 other than DoD, it is classified as "Restricted Computer Software" and the
 Government's rights in the Software are defined in paragraph 52.227-19 of the
 Federal Acquisition Regulations ("FAR") (or any successor regulations) or, in
 the cases of NASA, in paragraph 18.52.227-86 of the NASA Supplement to the FAR
 (or any successor regulations).

 -----------------------------------------------------------------------------

 Commercial licensing and support of this software is available from OpenSS7
 Corporation at a fee.  See http://www.openss7.com/

 -----------------------------------------------------------------------------

 Last Modified 2007/03/25 19:02:29 by brian

 *****************************************************************************/

#ident "@(#) sad.c,v openss7-0_9_2_F(0.9.2.40) 2007/03/25 19:02:29"

static char const ident[] =
    "sad.c,v openss7-0_9_2_F(0.9.2.40) 2007/03/25 19:02:29";

/*
 * STREAMS Administrative Driver (SAD) for Linux Fast-STREAMS.  Note that this driver also acts as a
 * pretty good example on how to write a general STREAMS pseudo-device driver.  Also, this driver
 * internally handles 32-bit to 64-bit conversions for compatible 32-bit ioctls.
 *
 * There are really two approaches to handling 32-bit to 64-bit conversions for ioctls: (1) register
 * a conversion function with the Stream head.  The conversion function registered converts the
 * 32-bit ioctl into a 64-bit iotcl and back again.  (2) recognize the IOC_ILP32 ioc_flag in the
 * iocblk structure indicating that the contents of the b_cont message block contains information
 * generated by a 32-bit ioctl call instead of a 64-bit one.
 *
 * The SAD driver demonstrates the later approach.  For kernels prior to 2.6.11, the later approach
 * requires that the ioctls be known by the Stream head and registered with the kernel for ioctl32
 * conversion (otherwise the 32-bit ioctl will never be passed to the driver).  For kernels after
 * 2.6.11, all ioctls generated by 32-bit ioctl calls will be passed to the driver.  SAD ioctls work
 * for both because they are known to the Stream head.
 */

#define _LFS_SOURCE
#include <sys/os7/compat.h>

#ifdef LIS
#define CONFIG_STREAMS_SAD_MODID	SAD_DRV_ID
#define CONFIG_STREAMS_SAD_NAME		SAD_DRV_NAME
#define CONFIG_STREAMS_SAD_MAJOR	SAD_CMAJOR_0
#endif

#define SAD_DESCRIP	"UNIX SYSTEM V RELEASE 4.2 FAST STREAMS FOR LINUX"
#define SAD_COPYRIGHT	"Copyright (c) 1997-2006 OpenSS7 Corporation.  All Rights Reserved."
#define SAD_REVISION	"LfS sad.c,v openss7-0_9_2_F(0.9.2.40) 2007/03/25 19:02:29"
#define SAD_DEVICE	"SVR 4.2 STREAMS Administrative Driver (SAD)"
#define SAD_CONTACT	"Brian Bidulock <bidulock@openss7.org>"
#define SAD_LICENSE	"GPL"
#define SAD_BANNER	SAD_DESCRIP	"\n" \
			SAD_COPYRIGHT	"\n" \
			SAD_REVISION	"\n" \
			SAD_DEVICE	"\n" \
			SAD_CONTACT	"\n"
#define SAD_SPLASH	SAD_DEVICE	" - " \
			SAD_REVISION	"\n"

#ifdef CONFIG_STREAMS_SAD_MODULE
MODULE_AUTHOR(SAD_CONTACT);
MODULE_DESCRIPTION(SAD_DESCRIP);
MODULE_SUPPORTED_DEVICE(SAD_DEVICE);
MODULE_LICENSE(SAD_LICENSE);
#if defined MODULE_ALIAS
MODULE_ALIAS("streams-sad");
#endif
#endif

#ifndef CONFIG_STREAMS_SAD_NAME
#error CONFIG_STREAMS_SAD_NAME must be defined.
#endif
#ifndef CONFIG_STREAMS_SAD_MAJOR
#error CONFIG_STREAMS_SAD_MAJOR must be defined.
#endif
#ifndef CONFIG_STREAMS_SAD_MODID
#error CONFIG_STREAMS_SAD_MODID must be defined.
#endif

modID_t modid = CONFIG_STREAMS_SAD_MODID;

#ifndef module_param
MODULE_PARM(modid, "h");
#else
module_param(modid, ushort, 0444);
#endif
MODULE_PARM_DESC(modid, "Module id number for STREAMS-administrative driver.");

#ifdef MODULE_ALIAS
MODULE_ALIAS("streams-modid-" __stringify(CONFIG_STREAMS_SAD_MODID));
MODULE_ALIAS("streams-driver-sad");
#endif

major_t major = CONFIG_STREAMS_SAD_MAJOR;

#ifndef module_param
MODULE_PARM(major, "h");
#else
module_param(major, uint, 0444);
#endif
MODULE_PARM_DESC(major, "Major device number for STREAMS-administrative driver.");

#ifdef MODULE_ALIAS
MODULE_ALIAS("char-major-" __stringify(CONFIG_STREAMS_SAD_MAJOR) "-*");
MODULE_ALIAS("/dev/sad");
MODULE_ALIAS("/dev/sad/admin");
MODULE_ALIAS("/dev/sad/user");
#ifdef LFS
MODULE_ALIAS("streams-major-" __stringify(CONFIG_STREAMS_SAD_MAJOR));
MODULE_ALIAS("/dev/streams/sad");
MODULE_ALIAS("/dev/streams/sad/*");
MODULE_ALIAS("/dev/streams/clone/sad");
MODULE_ALIAS("/dev/streams/sad/admin");
MODULE_ALIAS("/dev/streams/sad/user");
#endif
#endif

static struct module_info sad_minfo = {
	.mi_idnum = CONFIG_STREAMS_SAD_MODID,
	.mi_idname = CONFIG_STREAMS_SAD_NAME,
	.mi_minpsz = STRMINPSZ,
	.mi_maxpsz = STRMAXPSZ,
	.mi_hiwat = STRHIGH,
	.mi_lowat = STRLOW,
};

static struct module_stat sad_mstat __attribute__((__aligned__(SMP_CACHE_BYTES)));

#if defined __LP64__ && defined LFS
#  undef WITH_32BIT_CONVERSION
#  define WITH_32BIT_CONVERSION 1
#endif

/* private structures */
struct sad {
	int assigned;
	int iocstate;
	int transparent;
	struct str_list sl;
} sads[2];

#ifdef WITH_32BIT_CONVERSION
static void
sap32_convert(struct strapush32 *sap32, struct strapush *sap)
{
	sap->sap_cmd = sap32->sap_cmd;
	sap->sap_major = sap32->sap_major;
	sap->sap_minor = sap32->sap_minor;
	sap->sap_lastminor = sap32->sap_lastminor;
	sap->sap_npush = sap32->sap_npush;
	bcopy(&sap32->sap_list, &sap->sap_list, sizeof(sap->sap_list));
	sap->sap_anchor = sap32->sap_anchor;
	bcopy(&sap32->sap_module, &sap->sap_module, sizeof(sap->sap_module));
}
static void
sap32_revert(struct strapush *sap, struct strapush32 *sap32)
{
	sap32->sap_cmd = sap->sap_cmd;
	sap32->sap_major = sap->sap_major;
	sap32->sap_minor = sap->sap_minor;
	sap32->sap_lastminor = sap->sap_lastminor;
	sap32->sap_npush = sap->sap_npush;
	bcopy(&sap->sap_list, &sap32->sap_list, sizeof(sap->sap_list));
	sap32->sap_anchor = sap->sap_anchor;
	bcopy(&sap->sap_module, &sap32->sap_module, sizeof(sap->sap_module));
}
#endif

static streamscall int
sad_put(queue_t *q, mblk_t *mp)
{
	struct sad *sad = q->q_ptr;
	union ioctypes *ioc = (typeof(ioc)) mp->b_rptr;
	int err = 0, rval = 0, count = 0;
	mblk_t *dp = mp->b_cont;
	caddr_t sa_addr, sl_addr;
	size_t sa_size, sl_size;

	switch (mp->b_datap->db_type) {
	case M_FLUSH:
		if (mp->b_rptr[0] & FLUSHW) {
			if (mp->b_rptr[0] & FLUSHBAND)
				flushband(q, mp->b_rptr[1], FLUSHDATA);
			else
				flushq(q, FLUSHDATA);
			mp->b_rptr[0] &= ~FLUSHW;
		}
		if (mp->b_rptr[0] & FLUSHR) {
			queue_t *rq = RD(q);

			if (mp->b_rptr[0] & FLUSHBAND)
				flushband(rq, mp->b_rptr[1], FLUSHDATA);
			else
				flushq(rq, FLUSHDATA);
			qreply(q, mp);
			return (0);
		}
		break;
	case M_IOCTL:
		ioc = (typeof(ioc)) mp->b_rptr;
#ifdef WITH_32BIT_CONVERSION
		if (ioc->iocblk.ioc_flag == IOC_ILP32) {
			/* XXX: following pointer conversion does not work on all architectures. */
			sa_addr =
			    (caddr_t) (unsigned long) (uint32_t) *(unsigned long *) dp->b_rptr;
			sa_size = sizeof(struct strapush32);
			sl_addr = sa_addr;
			sl_size = sizeof(struct str_list32);
		} else
#endif
		{
			sa_addr = (caddr_t) *(unsigned long *) dp->b_rptr;
			sa_size = sizeof(struct strapush);
			sl_addr = sa_addr;
			sl_size = sizeof(struct str_list);
		}
		switch (ioc->iocblk.ioc_cmd) {
		case SAD_SAP:
			err = -EPERM;
			if (ioc->iocblk.ioc_uid != 0)
				goto nak;
			if (ioc->iocblk.ioc_count == TRANSPARENT) {
				mp->b_datap->db_type = M_COPYIN;
				ioc->copyreq.cq_addr = sa_addr;
				ioc->copyreq.cq_size = sa_size;
				ioc->copyreq.cq_flag = 0;
				ioc->copyreq.cq_private = (mblk_t *) ioc->copyreq.cq_addr;
				sad->transparent = 1;
				sad->iocstate = 1;
				qreply(q, mp);
				return (0);
			}
			sad->transparent = 0;
			sad->iocstate = 1;
			goto sad_sap_state1;
		case SAD_GAP:
			if (ioc->iocblk.ioc_count == TRANSPARENT) {
				mp->b_datap->db_type = M_COPYIN;
				ioc->copyreq.cq_addr = sa_addr;
				ioc->copyreq.cq_size = sa_size;
				ioc->copyreq.cq_flag = 0;
				ioc->copyreq.cq_private = (mblk_t *) ioc->copyreq.cq_addr;
				sad->transparent = 1;
				sad->iocstate = 1;
				qreply(q, mp);
				return (0);
			}
			sad->transparent = 0;
			sad->iocstate = 1;
			goto sad_gap_state1;
		case SAD_VML:
			if (ioc->iocblk.ioc_count == TRANSPARENT) {
				mp->b_datap->db_type = M_COPYIN;
				ioc->copyreq.cq_addr = sl_addr;
				ioc->copyreq.cq_size = sl_size;
				ioc->copyreq.cq_flag = 0;
				ioc->copyreq.cq_private = (mblk_t *) ioc->copyreq.cq_addr;
				sad->transparent = 1;
				sad->iocstate = 1;
				qreply(q, mp);
				return (0);
			}
			sad->transparent = 0;
			sad->iocstate = 1;
			goto sad_vml_state1;
		}
		err = -EINVAL;
		goto nak;
	case M_IOCDATA:
		ioc = (typeof(ioc)) mp->b_rptr;
		if (ioc->copyresp.cp_rval != (caddr_t) 0) {
			sad->transparent = 0;
			sad->iocstate = 0;
#ifdef LFS
			goto abort;
#endif
#ifdef LIS
			/* LiS has a bug here... */
			err = -(long) ioc->copyresp.cp_rval;
			goto nak;
#endif
		}
#ifdef WITH_32BIT_CONVERSION
		if (ioc->copyresp.cp_flag == IOC_ILP32) {
			sa_size = sizeof(struct strapush32);
			sl_size = sizeof(struct str_list32);
		} else
#endif
		{
			sa_size = sizeof(struct strapush);
			sl_size = sizeof(struct str_list);
		}
		switch (ioc->copyresp.cp_cmd) {
		case SAD_SAP:
			switch (sad->iocstate) {
			case 1:
			      sad_sap_state1:
				err = -EFAULT;
				if (!dp || dp->b_wptr < dp->b_rptr + sa_size)
					goto nak;
#ifdef WITH_32BIT_CONVERSION
				if (ioc->copyresp.cp_flag == IOC_ILP32) {
					struct strapush32 *sap32 = (typeof(sap32)) dp->b_rptr;
					struct strapush sa, *sap = &sa;

					sap32_convert(sap32, sap);
					if ((err = apush_set(sap)))
						goto nak;
					sap32_revert(sap, sap32);
				} else
#endif
				{
					struct strapush *sap = (typeof(sap)) dp->b_rptr;

					if ((err = apush_set(sap)))
						goto nak;
				}
				if (sad->transparent == 1) {
					mp->b_datap->db_type = M_COPYOUT;
					ioc->copyreq.cq_addr = (caddr_t) ioc->copyresp.cp_private;
					ioc->copyreq.cq_size = sa_size;
					ioc->copyreq.cq_flag = 0;
					sad->transparent = 1;
					sad->iocstate = 2;
					qreply(q, mp);
					return (0);
				}
				/* use implied I_STR copyout */
				count = sa_size;
				goto ack;
			case 2:
				/* done */
				goto ack;
			}
			err = -EIO;
			goto nak;
		case SAD_GAP:
			switch (sad->iocstate) {
			case 1:
			      sad_gap_state1:
				err = -EFAULT;
				if (!dp || dp->b_wptr < dp->b_rptr + sa_size)
					goto nak;
#ifdef WITH_32BIT_CONVERSION
				if (ioc->copyresp.cp_flag == IOC_ILP32) {
					struct strapush32 *sap32 = (typeof(sap32)) dp->b_rptr;
					struct strapush sa, *sap = &sa;

					sap32_convert(sap32, sap);
					if ((err = apush_get(sap)))
						goto nak;
					sap32_revert(sap, sap32);
				} else
#endif
				{
					struct strapush *sap;

					sap = (typeof(sap)) dp->b_rptr;
					if ((err = apush_get(sap)))
						goto nak;
				}
				if (sad->transparent == 1) {
					mp->b_datap->db_type = M_COPYOUT;
					ioc->copyreq.cq_addr = (caddr_t) ioc->copyresp.cp_private;
					ioc->copyreq.cq_size = sa_size;
					ioc->copyreq.cq_flag = 0;
					sad->transparent = 1;
					sad->iocstate = 2;
					qreply(q, mp);
					return (0);
				}
				/* use implied I_STR copyout */
				count = sa_size;
				goto ack;
			case 2:
				/* done */
				goto ack;
			}
			err = -EIO;
			goto nak;
		case SAD_VML:
			switch (sad->iocstate) {
			case 1:
			      sad_vml_state1:
				err = -EFAULT;
				if (!dp || dp->b_wptr < dp->b_rptr + sl_size)
					goto nak;
#ifdef WITH_32BIT_CONVERSION
				if (ioc->copyresp.cp_flag == IOC_ILP32) {
					struct str_list32 *slp32 = (typeof(slp32)) dp->b_rptr;

					sad->sl.sl_nmods = slp32->sl_nmods;
					sad->sl.sl_modlist =
					    (struct str_mlist *) (unsigned long) slp32->sl_modlist;
				} else
#endif
				{
					struct str_list *slp = (typeof(slp)) dp->b_rptr;

					sad->sl.sl_nmods = slp->sl_nmods;
					sad->sl.sl_modlist = slp->sl_modlist;
				}
				err = -EINVAL;
				if (1 > sad->sl.sl_nmods || sad->sl.sl_nmods > MAXAPUSH)
					goto nak;
				mp->b_datap->db_type = M_COPYIN;
				ioc->copyreq.cq_addr = (caddr_t) sad->sl.sl_modlist;
				ioc->copyreq.cq_size = sad->sl.sl_nmods * sizeof(struct str_mlist);
				ioc->copyreq.cq_flag = 0;
				sad->iocstate = 2;
				qreply(q, mp);
				return (0);
			case 2:
				err = -EFAULT;
				if (!dp || dp->b_wptr < dp->b_rptr
				    + sad->sl.sl_nmods * sizeof(struct str_mlist))
					goto nak;
				sad->sl.sl_modlist = (struct str_mlist *) dp->b_rptr;
				if ((err = apush_vml(&sad->sl)) < 0)
					goto nak;
				rval = err;
				goto ack;
			}
			err = -EIO;
			goto nak;
		}
	}
#ifdef LFS
      abort:
	freemsg(mp);
	return (0);
#endif
      nak:
	sad->iocstate = 0;
	mp->b_datap->db_type = M_IOCNAK;
	ioc->iocblk.ioc_count = 0;
	ioc->iocblk.ioc_rval = -1;
	ioc->iocblk.ioc_error = -err;
	sad->transparent = 0;
	sad->iocstate = 0;
	qreply(q, mp);
	return (0);
      ack:
	sad->iocstate = 0;
	mp->b_datap->db_type = M_IOCACK;
	ioc->iocblk.ioc_count = count;
	ioc->iocblk.ioc_rval = rval;
	ioc->iocblk.ioc_error = 0;
	sad->transparent = 0;
	sad->iocstate = 0;
	qreply(q, mp);
	return (0);
}

 /* There are two minors 0 is the /dev/sad/admin driver and 1 is the /dev/sad/user driver.
    Permission for access to the /dev/sad/admin minor is performed by filesystem permission on the
    character device and a check on open. */
static streamscall int
sad_open(queue_t *q, dev_t *devp, int oflag, int sflag, cred_t *crp)
{
	major_t major = getmajor(*devp);
	minor_t minor = getminor(*devp);

	if (q->q_ptr)
		return (0);	/* already open */
	switch (sflag) {
	case CLONEOPEN:
		minor = 1;
		if (sads[minor].assigned != 0)
			break;
	case DRVOPEN:
		if (minor != 0 && minor != 1)
			break;
		if (minor == 0 && crp->cr_uid != 0 && crp->cr_ruid != 0)
			return (-EACCES);
		*devp = makedevice(major, minor);
		sads[minor].assigned |= 1;
		q->q_ptr = WR(q)->q_ptr = &sads[minor];
		qprocson(q);
		return (0);
	}
	return (-ENXIO);
}
static streamscall int
sad_close(queue_t *q, int oflag, cred_t *crp)
{
	struct sad *sad = q->q_ptr;

	qprocsoff(q);
	q->q_ptr = WR(q)->q_ptr = NULL;
	sad->assigned = 0;
	sad->iocstate = 0;
	return (0);
}

static struct qinit sad_qinit = {
	.qi_putp = sad_put,
	.qi_qopen = sad_open,
	.qi_qclose = sad_close,
	.qi_minfo = &sad_minfo,
	.qi_mstat = &sad_mstat,
};

static struct streamtab sad_info = {
	.st_rdinit = &sad_qinit,
	.st_wrinit = &sad_qinit,
};

static struct cdevsw sad_cdev = {
	.d_name = "sad",
	.d_str = &sad_info,
	.d_flag = D_MP,
	.d_fop = NULL,
	.d_mode = S_IFCHR | S_IRUGO | S_IWUGO,
	.d_kmod = THIS_MODULE,
};

static struct devnode sad_node_admin = {
	.n_name = "admin",
	.n_flag = 0,
	.n_mode = S_IFCHR | S_IRUSR | S_IWUSR,
};

static struct devnode sad_node_user = {
	.n_name = "user",
	.n_flag = 0,
	.n_mode = S_IFCHR | S_IRUGO | S_IWUGO,
};

/*
 *  The following is an example of how to register IO control commands to be delivered by the Stream
 *  head to the module or driver.  Note that this is only required for Linux kernels prior 2.6.11.
 */
struct sad_ioctl {
	unsigned int cmd;
	void *opaque;
};

static struct sad_ioctl sad_map[] = {
	{.cmd = SAD_SAP,}
	, {.cmd = SAD_GAP,}
	, {.cmd = SAD_VML,}
#ifdef LFS
	, {.cmd = SAD_SAP_SOL,}
	, {.cmd = SAD_GAP_SOL,}
#endif
	, {.cmd = 0,}
};

void
sad_unregister_ioctl32(void)
{
#ifdef LFS
	struct sad_ioctl *i;

	for (i = sad_map; i->cmd != 0; i++)
		if (i->opaque != NULL)
			unregister_ioctl32(i->opaque);
#endif
	(void) sad_map;
}

int
sad_register_ioctl32(void)
{
#ifdef LFS
	struct sad_ioctl *i;

	for (i = sad_map; i->cmd != 0; i++) {
		if ((i->opaque = register_ioctl32(i->cmd)) == NULL) {
			sad_unregister_ioctl32();
			return (-ENOMEM);
		}
	}
#endif
	(void) sad_map;
	return (0);
}

#ifdef CONFIG_STREAMS_SAD_MODULE
static
#endif
int __init
sad_init(void)
{
	int err;

#ifdef CONFIG_STREAMS_SAD_MODULE
	printk(KERN_INFO SAD_BANNER);
#else
	printk(KERN_INFO SAD_SPLASH);
#endif
	if ((err = sad_register_ioctl32()) < 0)
		return (err);
	sad_minfo.mi_idnum = modid;
	if ((err = register_strdev(&sad_cdev, major)) < 0) {
		sad_unregister_ioctl32();
		return (err);
	}
	if (major == 0 && err > 0)
		major = err;
	bzero(&sads, sizeof(sads));
	register_strnod(&sad_cdev, &sad_node_admin, 0);
	register_strnod(&sad_cdev, &sad_node_user, 1);
	return (0);
};

#ifdef CONFIG_STREAMS_SAD_MODULE
static
#endif
void __exit
sad_exit(void)
{
	unregister_strnod(&sad_cdev, 1);
	unregister_strnod(&sad_cdev, 0);
	unregister_strdev(&sad_cdev, major);
	sad_unregister_ioctl32();
};

#ifdef CONFIG_STREAMS_SAD_MODULE
module_init(sad_init);
module_exit(sad_exit);
#endif
